//
//  ActuarialModel.swift
//  LifeClock
//
//  Created by Task-Master on 2025-09-16.
//

import Foundation

// MARK: - Health Metrics Structure
/// Standardized health data input structure for actuarial calculations

// MARK: - Supporting Enums


// MARK: - Calculation Results
/// Result structure for actuarial calculations
public struct ActuarialResult {
    let lifeExpectancyYears: Double
    let confidenceInterval: ConfidenceInterval
    let riskFactorImpacts: [RiskFactorImpact]
    let calculationDate: Date
    let methodUsed: String
    
    public init(lifeExpectancyYears: Double, 
                confidenceInterval: ConfidenceInterval,
                riskFactorImpacts: [RiskFactorImpact] = [],
                calculationDate: Date = Date(),
                methodUsed: String) {
        self.lifeExpectancyYears = lifeExpectancyYears
        self.confidenceInterval = confidenceInterval
        self.riskFactorImpacts = riskFactorImpacts
        self.calculationDate = calculationDate
        self.methodUsed = methodUsed
    }
}

public struct ConfidenceInterval {
    let lowerBound: Double
    let upperBound: Double
    let confidenceLevel: Double // e.g., 0.95 for 95%
    
    public init(lowerBound: Double, upperBound: Double, confidenceLevel: Double = 0.95) {
        self.lowerBound = lowerBound
        self.upperBound = upperBound
        self.confidenceLevel = confidenceLevel
    }
}

public struct RiskFactorImpact {
    let factor: String
    let yearsImpact: Double // positive = increases life expectancy, negative = decreases
    let hazardRatio: Double
    
    public init(factor: String, yearsImpact: Double, hazardRatio: Double) {
        self.factor = factor
        self.yearsImpact = yearsImpact
        self.hazardRatio = hazardRatio
    }
}

// MARK: - Error Handling
public enum ActuarialError: Error, LocalizedError {
    case invalidAge(String)
    case invalidHealthMetric(String)
    case insufficientData(String)
    case calculationError(String)
    case modelNotInitialized(String)
    
    public var errorDescription: String? {
        switch self {
        case .invalidAge(let message): return "Invalid age: \(message)"
        case .invalidHealthMetric(let message): return "Invalid health metric: \(message)"
        case .insufficientData(let message): return "Insufficient data: \(message)"
        case .calculationError(let message): return "Calculation error: \(message)"
        case .modelNotInitialized(let message): return "Model not initialized: \(message)"
        }
    }
}

// MARK: - Mortality Model Protocol
/// Protocol defining required methods for mortality model implementations
public protocol MortalityModel {
    /// Calculate mortality rate at given age
    /// - Parameters:
    ///   - age: Current age in years
    ///   - healthMetrics: Optional health metrics for adjustment
    /// - Returns: Annual mortality rate (probability of death in next year)
    func mortalityRate(at age: Double, healthMetrics: HealthMetrics?) throws -> Double
    
    /// Calculate survival probability to given age
    /// - Parameters:
    ///   - currentAge: Current age in years
    ///   - targetAge: Target age for survival probability
    ///   - healthMetrics: Optional health metrics for adjustment
    /// - Returns: Probability of surviving to target age
    func survivalProbability(from currentAge: Double, to targetAge: Double, healthMetrics: HealthMetrics?) throws -> Double
    
    /// Calculate life expectancy
    /// - Parameters:
    ///   - age: Current age in years
    ///   - healthMetrics: Optional health metrics for adjustment
    /// - Returns: Expected remaining years of life
    func lifeExpectancy(at age: Double, healthMetrics: HealthMetrics?) throws -> Double
    
    /// Model identifier for logging and debugging
    var modelName: String { get }
    
    /// Model version for tracking updates
    var modelVersion: String { get }
}

// MARK: - Base Actuarial Calculator
/// Base class providing common functionality for actuarial calculations
public class ActuarialCalculator {
    private let mortalityModel: MortalityModel
    private var isInitialized: Bool = false
    
    public init(mortalityModel: MortalityModel) {
        self.mortalityModel = mortalityModel
        self.isInitialized = true
    }
    
    /// Main calculation method combining mortality model with health metrics
    /// - Parameters:
    ///   - age: Current age in years
    ///   - gender: Gender for demographic adjustments
    ///   - healthMetrics: Health metrics for risk adjustment
    /// - Returns: Complete actuarial result with confidence intervals
    public func calculateLifeExpectancy(age: Double, 
                                      gender: Gender, 
                                      healthMetrics: HealthMetrics) throws -> ActuarialResult {
        guard isInitialized else {
            throw ActuarialError.modelNotInitialized("Calculator must be initialized before use")
        }
        
        try validateInputs(age: age, healthMetrics: healthMetrics)
        
        // Base life expectancy from mortality model
        let baseLifeExpectancy = try mortalityModel.lifeExpectancy(at: age, healthMetrics: healthMetrics)
        
        // Calculate risk factor adjustments
        let riskFactorImpacts = calculateRiskFactorImpacts(healthMetrics: healthMetrics, baseLifeExpectancy: baseLifeExpectancy)
        
        // Apply risk adjustments
        let adjustedLifeExpectancy = applyRiskAdjustments(
            baseLifeExpectancy: baseLifeExpectancy,
            riskFactorImpacts: riskFactorImpacts
        )
        
        // Calculate confidence intervals (simplified approach)
        let confidenceInterval = calculateConfidenceInterval(
            lifeExpectancy: adjustedLifeExpectancy,
            age: age,
            healthMetrics: healthMetrics
        )
        
        return ActuarialResult(
            lifeExpectancyYears: adjustedLifeExpectancy,
            confidenceInterval: confidenceInterval,
            riskFactorImpacts: riskFactorImpacts,
            calculationDate: Date(),
            methodUsed: mortalityModel.modelName
        )
    }
    
    // MARK: - Private Helper Methods
    
    private func validateInputs(age: Double, healthMetrics: HealthMetrics) throws {
        guard age >= 0 && age <= 150 else {
            throw ActuarialError.invalidAge("Age must be between 0 and 150 years")
        }
        
        try healthMetrics.validate()
    }
    
    private func calculateRiskFactorImpacts(healthMetrics: HealthMetrics, baseLifeExpectancy: Double) -> [RiskFactorImpact] {
        var impacts: [RiskFactorImpact] = []
        
        // Smoking impact
        let smokingHR = healthMetrics.smokingStatus.hazardRatio
        if smokingHR != 1.0 {
            let yearsImpact = calculateYearsFromHazardRatio(hazardRatio: smokingHR, baseLifeExpectancy: baseLifeExpectancy)
            impacts.append(RiskFactorImpact(factor: "Smoking", yearsImpact: yearsImpact, hazardRatio: smokingHR))
        }
        
        // Exercise impact
        let exerciseHR = healthMetrics.exerciseFrequency.hazardRatio
        if exerciseHR != 1.0 {
            let yearsImpact = calculateYearsFromHazardRatio(hazardRatio: exerciseHR, baseLifeExpectancy: baseLifeExpectancy)
            impacts.append(RiskFactorImpact(factor: "Exercise", yearsImpact: yearsImpact, hazardRatio: exerciseHR))
        }
        
        // BMI impact (simplified)
        if let bmi = healthMetrics.bmi {
            let bmiHR = calculateBMIHazardRatio(bmi: bmi)
            if bmiHR != 1.0 {
                let yearsImpact = calculateYearsFromHazardRatio(hazardRatio: bmiHR, baseLifeExpectancy: baseLifeExpectancy)
                impacts.append(RiskFactorImpact(factor: "BMI", yearsImpact: yearsImpact, hazardRatio: bmiHR))
            }
        }
        
        return impacts
    }
    
    private func calculateBMIHazardRatio(bmi: Double) -> Double {
        // Simplified BMI hazard ratio calculation
        // Optimal BMI range: 20-25
        if bmi >= 20 && bmi <= 25 {
            return 1.0
        } else if bmi < 18.5 {
            return 1.2 // Underweight
        } else if bmi <= 30 {
            return 1.1 // Overweight
        } else if bmi <= 35 {
            return 1.3 // Obese Class I
        } else {
            return 1.6 // Obese Class II+
        }
    }
    
    private func calculateYearsFromHazardRatio(hazardRatio: Double, baseLifeExpectancy: Double) -> Double {
        // Simplified conversion: HR > 1 reduces life expectancy, HR < 1 increases it
        // This is a rough approximation and should be refined with proper actuarial methods
        let logHR = log(hazardRatio)
        return -logHR * (baseLifeExpectancy * 0.1) // Scale factor for impact
    }
    
    private func applyRiskAdjustments(baseLifeExpectancy: Double, riskFactorImpacts: [RiskFactorImpact]) -> Double {
        let totalAdjustment = riskFactorImpacts.reduce(0) { $0 + $1.yearsImpact }
        return max(0.1, baseLifeExpectancy + totalAdjustment) // Ensure positive result
    }
    
    private func calculateConfidenceInterval(lifeExpectancy: Double, age: Double, healthMetrics: HealthMetrics) -> ConfidenceInterval {
        // Simplified confidence interval calculation
        // In practice, this would use statistical methods based on model uncertainty
        let standardError = lifeExpectancy * 0.1 // Assume 10% standard error
        let marginOfError = standardError * 1.96 // 95% confidence interval
        
        return ConfidenceInterval(
            lowerBound: max(0, lifeExpectancy - marginOfError),
            upperBound: lifeExpectancy + marginOfError,
            confidenceLevel: 0.95
        )
    }
}

// MARK: - Supporting Types

// MARK: - Model Registry
/// Registry for managing different mortality models
public class MortalityModelRegistry {
    private static var registeredModels: [String: MortalityModel] = [:]
    
    public static func register(model: MortalityModel) {
        registeredModels[model.modelName] = model
    }
    
    public static func getModel(named: String) -> MortalityModel? {
        return registeredModels[named]
    }
    
    public static func availableModels() -> [String] {
        return Array(registeredModels.keys)
    }
}